using UnityEngine;
using System;
using System.Runtime.InteropServices;
using System.Text;

namespace Sony
{
	namespace NP
	{
		public class User
		{
			// Signing In.
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSignin();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxSigninIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxIsSignedIn();

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxSigninGetLastError(out ResultCode result);

			[DllImport("UnityNpToolkit")]
			private static extern int PrxSetCurrentUserId(int UserId);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetCurrentUserId();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxLogOutUser(int UserId);

			public static int SetCurrentUserId(int UserId)
			{
				return PrxSetCurrentUserId(UserId);
			}

			public static int GetCurrentUserId()
			{
				return PrxGetCurrentUserId();
			}

			public static int LogOutUser(int UserId)
			{
				return PrxLogOutUser(UserId);
			}


			public static bool GetLastSignInError(out ResultCode result)
			{
				PrxSigninGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxUserProfileGetLastError(out ResultCode result);
			
			public static bool GetLastUserProfileError(out ResultCode result)
			{
				PrxUserProfileGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}
			
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxPresenceGetLastError(out ResultCode result);
			
			public static bool GetLastPresenceError(out ResultCode result)
			{
				PrxPresenceGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			// Online Presence.
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxSetPresence(string text);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxSetPresenceIsBusy();

			// User Profile.
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxRequestUserProfile();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxUserProfileIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxGetUserProfile(out UserProfile profile);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetUserSigninStatus(int userID, ref byte[] npID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxRequestRemoteUserNpID(string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxGetRemoteUserNpID(out NpID npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxRequestRemoteUserProfileForOnlineID(string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxRequestRemoteUserProfileForNpID(byte[] npID);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxGetRemoteUserProfile(out RemoteUserProfile profile);

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct UserProfile
			{
				public int language;
				public int age;
				public int chatRestricted;
				public int contentRestricted;
				IntPtr _onlineID;
				IntPtr _npID;
				int npIDSize;
				IntPtr _avatarURL;
				IntPtr _countryCode;
				IntPtr _firstName;
				IntPtr _middleName;
				IntPtr _lastName;
				IntPtr _profilePictureUrl;
				public ulong npAccountId;
				public string firstName { get { return Marshal.PtrToStringAnsi(_firstName); } }
				public string middleName { get { return Marshal.PtrToStringAnsi(_middleName); } }
				public string lastName { get { return Marshal.PtrToStringAnsi(_lastName); } }
				public string profilePictureUrl { get { return Marshal.PtrToStringAnsi(_profilePictureUrl); } }
				public string onlineID { get { return Marshal.PtrToStringAnsi(_onlineID); } }

				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
				public string avatarURL { get { return Marshal.PtrToStringAnsi(_avatarURL); } }
				public string countryCode { get { return Marshal.PtrToStringAnsi(_countryCode); } }
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct NpID
			{
				IntPtr _npID;
				int npIDSize;

				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
			};
			
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct RemoteUserProfile
			{
				public int language;
				IntPtr _onlineID;
				IntPtr _npID;
				int npIDSize;
				IntPtr _avatarURL;
				IntPtr _countryCode;
				IntPtr _firstName;
				IntPtr _middleName;
				IntPtr _lastName;
				IntPtr _profilePictureUrl;
				public string onlineID { get { return Marshal.PtrToStringAnsi(_onlineID); } }
				public string firstName { get { return Marshal.PtrToStringAnsi(_firstName); } }
				public string middleName { get { return Marshal.PtrToStringAnsi(_middleName); } }
				public string lastName { get { return Marshal.PtrToStringAnsi(_lastName); } }
				public string profilePictureUrl { get { return Marshal.PtrToStringAnsi(_profilePictureUrl); } }
				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
				public string avatarURL { get { return Marshal.PtrToStringAnsi(_avatarURL); } }
				public string countryCode { get { return Marshal.PtrToStringAnsi(_countryCode); } }
			};

			public static event Messages.EventHandler OnSignedIn;
			public static event Messages.EventHandler OnSignedOut;
			public static event Messages.EventHandler OnPresenceSet;
			public static event Messages.EventHandler OnGotUserProfile;
			public static event Messages.EventHandler OnGotRemoteUserNpID;
			public static event Messages.EventHandler OnGotRemoteUserProfile;
			public static event Messages.EventHandler OnUserProfileError;
			public static event Messages.EventHandler OnPresenceError;
			public static event Messages.EventHandler OnSignInError;

			public static bool signedIn = false;

			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_SignedIn:
						signedIn = true;
						System.connectionUp = true;	//	By definition if we get a signed in message, we must be connected
						if (OnSignedIn != null) OnSignedIn(msg);
						return true;

					case Messages.MessageType.kNPToolKit_SignedOut:
						signedIn = false;
						if (OnSignedOut != null) OnSignedOut(msg);
						return true;

					case Messages.MessageType.kNPToolKit_PresenceSet:
						if (OnPresenceSet != null) OnPresenceSet(msg);
						return true;

					case Messages.MessageType.kNPToolKit_GotUserProfile:
						if (OnGotUserProfile != null) OnGotUserProfile(msg);
						return true;

					case Messages.MessageType.kNPToolKit_GotRemoteUserProfile:
						if (OnGotRemoteUserProfile != null) OnGotRemoteUserProfile(msg);
						return true;

					case Messages.MessageType.kNPToolKit_GotRemoteUserNpID:
						if (OnGotRemoteUserNpID != null) OnGotRemoteUserNpID(msg);
						return true;

					case Messages.MessageType.kNPToolKit_UserProfileError:
						if (OnUserProfileError != null)
						{
							OnUserProfileError(msg);
						}
						// on ps4 we check and error specific critial error code that aren't reported elsewhere
						if (Application.platform == RuntimePlatform.PS4)
						{
							ResultCode result = new ResultCode();
							GetLastUserProfileError(out result);
							switch ((UInt32)result.lastErrorSCE)
							{
								case 0x82200284:		// this happens if i use CUSA00702_00
								case 0x82200182:		// this seems to happen if i edit the title secret
									Debug.LogError(String.Format("bad missing NP title ID/ NP Title secret ... check your publishing settings ({0:X})", result.lastErrorSCE));
									break;
							}

						}

						return true;

					case Messages.MessageType.kNPToolKit_PresenceError:
						if (OnPresenceError != null) OnPresenceError(msg);
						break;

					case Messages.MessageType.kNPToolKit_SignInError:
						if (OnSignInError != null) OnSignInError(msg);
						break;
				}

				return false;
			}

			public static bool IsSignedInPSN
			{
				get
				{
					if (Application.platform == RuntimePlatform.PS4)
					{
						// On PS4 we get the current users PSN sign in state.
						byte[] npid;
						int status = Sony.NP.User.GetUserSigninStatus(Sony.NP.User.GetCurrentUserId(), out npid);
						return status == 2;
					}
					else
					{
						// On Vita & PS3 we only have one user and the PSN sign in state is indicated by Sony.NP.User.IsSignedIn.
						return Sony.NP.User.IsSignedIn;
					}
				}
			}

			public static bool IsSignedIn
			{
				get { return signedIn; }
			}

			public static bool IsSigninBusy
			{
				get { return PrxSigninIsBusy(); }
			}

			public static bool IsUserProfileBusy
			{
				get { return PrxUserProfileIsBusy(); }
			}

			public static ErrorCode SignIn()
			{
				return PrxSignin();
			}

			public static ErrorCode SetOnlinePresence(string text)
			{
				return PrxSetPresence(text);
			}

			// TODO: SetOnlineData(byte[] buffer)
			//{
			//}

			public static bool OnlinePresenceIsBusy()
			{
				return PrxSetPresenceIsBusy();
			}

			public static ErrorCode RequestUserProfile()
			{
				return PrxRequestUserProfile();
			}
			
			public static UserProfile GetCachedUserProfile()
			{
				UserProfile profile = new UserProfile();
				PrxGetUserProfile(out profile);
				return profile;
			}

			public static ErrorCode RequestRemoteUserNpID(string onlineID)
			{
				return PrxRequestRemoteUserNpID(onlineID);
			}


			public static int GetUserSigninStatus(int userID, out byte[] npID)
			{
				npID = new byte[36];
				npID[0] = 0xaa;
				npID[1] = 0xbb;

				return PrxGetUserSigninStatus(userID, ref npID);
			}

			public static byte[] GetCachedRemoteUserNpID()
			{
				NpID npID = new NpID();
				PrxGetRemoteUserNpID(out npID);
				return npID.npID;
			}

			public static ErrorCode RequestRemoteUserProfileForOnlineID(string onlineID)
			{
				return PrxRequestRemoteUserProfileForOnlineID(onlineID);
			}

			public static ErrorCode RequestRemoteUserProfileForNpID(byte[] npID)
			{
				return PrxRequestRemoteUserProfileForNpID(npID);
			}

			public static RemoteUserProfile GetCachedRemoteUserProfile()
			{
				RemoteUserProfile profile = new RemoteUserProfile();
				PrxGetRemoteUserProfile(out profile);
				return profile;
			}

			public static bool UserProfileIsBusy()
			{
				return PrxUserProfileIsBusy();
			}
		}
	} // NP
} // Sony
